#ifndef __LLF__KMNG_H__
#define __LLF__KMNG_H__


/*
   *  Object %name    : LLFKmng.h
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 17:39:24 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief  Key Management h file (API and structures )
   *
   *  \version 
   *  \author yevgenys
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */


/*------------------------------
    DEFINES
--------------------------------*/

/*-------------------------------
  STRUCTURES
---------------------------------*/
typedef struct KMNG_KeyDataParameter_t
{ 
  /* length of the buffer tha holds encrypted data */
  DxUint32_t  encryptedKeyDataLength;
  
  /* id of the key */
  DxUint32_t  keyId;
  
  /* length of the access password */
  DxUint32_t  keyPasswordLen;
  
  /* pointer to access password of the key */
  DxUint8_t*  keyPassword_ptr; 
  
  /* pointer to the encrypted data of the key */
  DxUint8_t*  encryptedKeyData_ptr;
  
}KMNG_KeyDataParameter_t;



/*------------------------------------------------
    FUNCTIONS
--------------------------------------------------*/


/**
 * @brief     This function initializes the LLF KMNG module
 *
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_Init(void);

/**
 * @brief     This function creates key accroding to the type, usage and restriction,
 *            and then wraps it and its data with the RKEK or the key passed as a parameter. The wrapped data is 
 *            loaded into the the provided memory. If the keyRingKeyData_ptr is NULL , then the created key is wrapped with RKEK
 *            The password and password lenght for the new key are passwed in the newKeyData_ptr.
 *			  On exit, newKeyData_ptr holds the created key data and key Id
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key ring key data
 * @param[in/out] newKeyData_ptr - lpointer to the created key data
 * @param[in] keyInfo_ptr - information of the created key
 * @param[in] KeyGenData_ptr - buffer for the internal use of the function (if buffer not need, NULL must be passed as argument)
 *
 * @return     DxError:  
 *                        
 */

DxError_t LLF_KMNG_CreateKey(KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
                             KMNG_KeyDataParameter_t *newKeyData_ptr,
                             KMNG_KeyConfiguration_t *keyConfiguration_ptr,
							               KMNG_BufferData_t		 *KeyGenData_ptr );
                          
/* create the key of the key ring */
#define LLF_KMNG_CreateKeyRingKey(newKeyData_ptr , keyConfiguration_ptr)\
		LLF_KMNG_CreateKey(DX_NULL , newKeyData_ptr , keyConfiguration_ptr , DX_NULL)

/* create user key */
#define LLF_KMNG_CreateUserKey(keyRingKeyData_ptr , newKeyData_ptr , keyConfiguration_ptr)\
		LLF_KMNG_CreateKey(keyRingKeyData_ptr , newKeyData_ptr , keyConfiguration_ptr , DX_NULL)                                

/* create user Symmetric key */
#define LLF_KMNG_CreateUserSymKey(keyRingKeyData_ptr , newKeyData_ptr , keyConfiguration_ptr)\
		LLF_KMNG_CreateKey(keyRingKeyData_ptr , newKeyData_ptr , keyConfiguration_ptr , DX_NULL) 

/* create user Asymmetric key */
#define LLF_KMNG_CreateUserAsymKey(keyRingKeyData_ptr , newKeyData_ptr , keyConfiguration_ptr, KeyGenData_ptr)\
		LLF_KMNG_CreateKey(keyRingKeyData_ptr , newKeyData_ptr , keyConfiguration_ptr , KeyGenData_ptr)                                


/**
 * @brief     This function imports a symmetric key into key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] keyConfiguration_ptr - configuration of the imported key
 * @param[in/out] importedKeyData_ptr - imported key data. On input , only password and password length fields are set
 * @param[in] importSymKey_ptr - symmtric key data to be imported
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ImportSymUserKey(KMNG_KeyDataParameter_t*    keyRingKeyData_ptr,
                                    KMNG_KeyConfiguration_t*    keyConfiguration_ptr,
                                    KMNG_KeyDataParameter_t*    importedKeyData_ptr,
                                    DxUint8_t*                  importSymKey_ptr);
                                       
                                                                       
/**
 * @brief     This function imports a RSA key into key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] keyConfiguration_ptr - configuration of the imported key
 * @param[in/out] importedKeyData_ptr - imported key data. On input , only password and password length fields are set
 * @param[in] modulus_ptr - modulus pointer
 * @param[in] modulusSizeInBytes - modulus size
 * @param[in] E_ptr - E pointer
 * @param[in] E_SizeInBytes - E size
 * @param[in] D_ptr - D pointer
 * @param[in] D_SizeInBytes - D size
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ImportRSAUserKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
                                   KMNG_KeyConfiguration_t*    keyConfiguration_ptr,
                                   KMNG_KeyDataParameter_t*   importedKeyData_ptr,
                                   DxUint8_t*                    modulus_ptr,
                                   DxUint32_t                    modulusSizeInBytes,
                                   DxUint8_t*                    E_ptr,
                                   DxUint32_t                    E_SizeInBytes,
                                   DxUint8_t*                    D_ptr,
                                   DxUint32_t                    D_SizeInBytes);
                                       

/**
 * @brief     This function imports a DH key into key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] keyConfiguration_ptr - configuration of the imported key
 * @param[in/out] importedKeyData_ptr - imported key data. On input , only password and password length fields are set
 * @param[in] modulus_ptr - modulus pointer
 * @param[in] modulusSize - modulus size
 * @param[in] generator_ptr - generator pointer
 * @param[in] generatorSize - generator size
 * @param[in] pub_ptr - pub key pointer
 * @param[in] pub_Size - pub key size in bytes
 * @param[in] priv_ptr - priv key pointer
 * @param[in] priv_Size - priv size in bytes
 * 
 * @return     DxError:  
 *                        
 */                                       
DxError_t LLF_KMNG_ImportDHUserKey(KMNG_KeyDataParameter_t*    keyRingKeyData_ptr,
                                KMNG_KeyConfiguration_t*     keyConfiguration_ptr,
                                KMNG_KeyDataParameter_t*    importedKeyData_ptr,
                                DxUint8_t*                     modulus_ptr,
                                DxUint32_t                     modulusSize,
                                DxUint8_t*                     generator_ptr,
                                DxUint32_t                     generatorSize,
                                DxUint8_t*                     pub_ptr,
                                DxUint32_t                     pub_Size,
                                DxUint8_t*                     priv_ptr,
                                DxUint32_t                     priv_Size);


/**
 * @brief     This function exports a symmetric key from key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[out] keyConfiguration_ptr - configuration of the exported key
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[out] keyData_ptr - symmetric key itself after decryption
 * @param[in] exportInternalFlag - if set, means that the export is done inside LLF implementation, and can be permitted even if restriction is non-exportable
 * 
 * @return     DxError:  
 *                        
 */                                       
DxError_t _LLF_KMNG_ExportSymUserKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
                                    KMNG_KeyConfiguration_t*   keyConfiguration_ptr,
                                    KMNG_KeyDataParameter_t*   exportKeyData_ptr,
                                    DxUint8_t*                 keyData_ptr,
									DxUint32_t*                keyDataLen_ptr,
                                    DxUint32_t                 exportInternalFlag);

/* export symmetric key */
#define LLF_KMNG_ExportSymUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , keyData_ptr , keyDataLen_ptr)\
    _LLF_KMNG_ExportSymUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , keyData_ptr , keyDataLen_ptr , 0) 

/* read symmetric key - this API must be called only from the SEP (LLF) layer */
#define LLF_KMNG_ReadSymUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , keyData_ptr , keyDataLen_ptr)\
    _LLF_KMNG_ExportSymUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , keyData_ptr , keyDataLen_ptr , 1) 


/**
 * @brief     This function exports an RSA key from key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[out] keyConfiguration_ptr - configuration of the exported key
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[out] modulus_ptr - modulus pointer
 * @param[out] modulusSize_ptr - modulus size
 * @param[out] E_ptr - N pointer
 * @param[out] E_Size_ptr - N size
 * @param[out] D_ptr - D pointer
 * @param[out] D_Size_ptr - D size
 * @param[in] exportInternalFlag - if set, means that the export is done inside LLF implementation, and can be permitted even if restriction is non-exportable
 * 
 * @return     DxError:  
 *                        
 */                                           
DxError_t _LLF_KMNG_ExportRSAUserKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
                                 KMNG_KeyConfiguration_t*      keyConfiguration_ptr,
                                 KMNG_KeyDataParameter_t*      exportKeyData_ptr,
                                 DxUint8_t*                    modulus_ptr,
                                 DxUint32_t*                   modulusSize_ptr,
                                 DxUint8_t*                    E_ptr,
                                 DxUint32_t*                   E_Size_ptr,
                                 DxUint8_t*                    D_ptr,
                                 DxUint32_t*                   D_Size_ptr,
                                 DxUint32_t                    exportInternalFlag);

/* export RSA key */
#define LLF_KMNG_ExportRSAUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , modulus_ptr , modulusSize_ptr , E_ptr , E_Size_ptr , D_ptr , D_Size_ptr)\
    _LLF_KMNG_ExportRSAUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , modulus_ptr , modulusSize_ptr , E_ptr , E_Size_ptr , D_ptr , D_Size_ptr , 0) 

/* read RSA key - this API must be called only from the SEP (LLF) layer */
#define LLF_KMNG_ReadRSAUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , modulus_ptr , modulusSize_ptr , E_ptr , E_Size_ptr , D_ptr , D_Size_ptr)\
    _LLF_KMNG_ExportRSAUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , modulus_ptr , modulusSize_ptr , E_ptr , E_Size_ptr , D_ptr , D_Size_ptr , 1) 


/**
 * @brief     This function exports an DH key from key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[out] keyConfiguration_ptr - configuration of the exported key
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[in] modulus_ptr - modulus pointer
 * @param[in] modulusSize - modulus size
 * @param[in] generator_ptr - generator pointer
 * @param[in] generatorSize_ptr - generator size in bytes
 * @param[in] pub_ptr - pub pointer
 * @param[in] pub_Size_ptr - pub size in bytes
 * @param[in] priv_ptr - private pointer
 * @param[in] priv_Size_ptr - private size in bytes
 * @param[in] exportInternalFlag - if set, means that the export is done inside LLF implementation, and can be permitted even if restriction is non-exportable
 *
 * @return     DxError:  
 *                        
 */                                           
DxError_t _LLF_KMNG_ExportDHUserKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
                                KMNG_KeyConfiguration_t*   keyConfiguration_ptr,
                                KMNG_KeyDataParameter_t*   exportKeyData_ptr,
                                DxUint8_t*                modulus_ptr,
                                DxUint32_t*               modulusSize_ptr,
                                DxUint8_t*                generator_ptr,
                                DxUint32_t*               generatorSize_ptr,
                                DxUint8_t*                pub_ptr,
                                DxUint32_t*               pub_Size_ptr,
                                DxUint8_t*                priv_ptr,
                                DxUint32_t*               priv_Size_ptr,
                                DxUint32_t                exportInternalFlag);

/* export DH key */
#define LLF_KMNG_ExportDHUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , modulus_ptr , modulusSize_ptr , generator_ptr , generatorSize_ptr , pub_ptr , pub_Size_ptr , priv_ptr , priv_Size_ptr)\
    _LLF_KMNG_ExportDHUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , modulus_ptr , modulusSize_ptr , generator_ptr , generatorSize_ptr , pub_ptr , pub_Size_ptr , priv_ptr , priv_Size_ptr , 0) 

/* read DH key - this API must be called only from the SEP (LLF) layer */
#define LLF_KMNG_ReadDHUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , modulus_ptr , modulusSize_ptr , generator_ptr , generatorSize_ptr , pub_ptr , pub_Size_ptr)\
    _LLF_KMNG_ExportDHUserKey(keyRingKeyData_ptr , keyConfiguration_ptr , exportKeyData_ptr , modulus_ptr , modulusSize_ptr , generator_ptr , generatorSize_ptr , pub_ptr , pub_Size_ptr , DX_NULL , DX_NULL , 1) 



/**
 * @brief     checks the authorization of the given key, by checking the hash of the password
 *            stored in the key data and the given password
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] authorizeKeyData_ptr - pointer to the data of the key to be authorized
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_AuthorizeKey(KMNG_KeyDataParameter_t*  keyRingKeyData_ptr,
                                KMNG_KeyDataParameter_t*  authorizeKeyData_ptr);
                                                                          
                                       
/**
 * @brief     This function transfers key from one key ring into another (actually decrypts it and then enctypts it with destination key ring key).
 *
 * @param[in] srcKeyRingKeyData_ptr - pointer to the key data of the source Key Ring
 * @param[in] srcKeyData_ptr - key data of the source key to be transfered
 * @param[in] destRingKeyData_ptr - pointer to the key data of the destination Key Ring
 * @param[out] destKeyData_ptr - data of key transferred to the destination key ring
 *
 * @return     DxError:  
 *                        
 */                                
DxError_t LLF_KMNG_TransferKey(KMNG_KeyDataParameter_t*  srcKeyRingKeyData_ptr,
                               KMNG_KeyDataParameter_t*  srcKeyData_ptr,
                               KMNG_KeyDataParameter_t*  destRingKeyData_ptr,
                               KMNG_KeyDataParameter_t*  destKeyData_ptr);
                                
                                
/**
 * @brief     This function prepares the AES contest for CRYS function usage based on the key ring AES key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]aesContext_ptr - memory to stores created AES context  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateAESKey(KMNG_KeyDataParameter_t	*keyRingKeyData_ptr,
								  KMNG_KeyDataParameter_t	*keyData_ptr,
								  KMNG_AES_WrappedKey_t		AES_WrappedKey_ptr);
                                          

/**
 * @brief     This function prepares the DES contest for CRYS function usage based on the key ring AES key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]aesContext_ptr - memory to stores created AES context  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateDESKey(KMNG_KeyDataParameter_t	*keyRingKeyData_ptr,
								  KMNG_KeyDataParameter_t	*keyData_ptr,
								  KMNG_DES_WrappedKey_t		DES_WrappedKey_ptr);
                                       
/**
 * @brief     This function prepares the HMAC contest for CRYS function usage based on the key ring HMAC key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]hmacContext_ptr - memory to stores created HMAC context  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateHMACKey(KMNG_KeyDataParameter_t	*keyRingKeyData_ptr,
								  KMNG_KeyDataParameter_t	*keyData_ptr,
								  KMNG_HMAC_WrappedKey_t	HMAC_WrappedKey_ptr);
                                             
/**
 * @brief     This function prepares the RSA contest for CRYS function usage based on the key ring RSA key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]rsaContext_ptr - memory to stores created RSA context  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateRSAKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
									  KMNG_KeyDataParameter_t*   keyData_ptr,
									  KMNG_RsaKeyOperationType_t rsaOperationType,
									  DxUint8_t*				RSA_WrappedKey_ptr,
									  DxUint32_t*				RSA_WrappedKeyLen_ptr);
                                    
                                    
/**
 * @brief     This function prepares the DH contest for CRYS function usage based on the key ring DH key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]dhContext_ptr - memory to stores created DH context  
 * @param[out]DH_WrappedKeyLen_ptr - the length of the wraped key 
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateDHKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
								 KMNG_KeyDataParameter_t*   keyData_ptr,
								 KMNG_DH_WrappedKey_t		DH_WrappedKey_ptr,
								 DxUint32_t*				DH_WrappedKeyLen_ptr); 
                                   

/**
 * @brief     this fucntion caluclates the AES mac on the given image buffer and compares it to the given MAC buffer 
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] imageBuffer_ptr - pointer to the image to be calculate on
 * @param[in] imageBufferSize - image buffer size
 * @param[out]MacBuffer_ptr - The AES MAC of the keyRing_ptr data  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_CalcAndVerifyKeyRingAesMAC(KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
                      											  DxUint8_t			*imageBuffer_ptr,
                      											  DxUint32_t		imageBufferSize,
                      											  DxUint8_t			*MacBuffer_ptr);

/**
 * @brief     this fucntion caluclates the AES mac on the given ke ring buffer 
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyRing_ptr - pointer to the image to be calculate on
 * @param[in] keyRingLength - image buffer size
 * @param[out]MacBuffer_ptr - The AES MAC of the keyRing_ptr data  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_CalcKeyRingAesMAC(KMNG_KeyDataParameter_t* keyRingDataParam_ptr,
                  									 DxUint8_t*               keyRing_ptr,
                  									 DxUint32_t               keyRingLength,
                  									 DxUint8_t*               macBuffer_ptr);
                    

/**
 * @brief     This function retrive the configuration of key from key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[out] keyConfiguration_ptr - configuration of the exported key
 * 
 * @return     DxError:  
 *                        
 */                                       
DxError_t LLF_KMNG_RetrieveKeyConfiguration(KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
											KMNG_KeyDataParameter_t *retrivedKeyData_ptr,
											KMNG_KeyConfiguration_t *keyConfiguration_ptr);

/**
 * @brief     This function updates the user specific data in the key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[in] UserSpecificKeyData - new user data
 * 
 * @return     DxError:  
 *                        
 */                                       
DxError_t LLF_KMNG_UpdateKeyUserData(KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
								  KMNG_KeyDataParameter_t *updateKeyData_ptr,
								   KMNG_UserSpecificKeyData_t   UserSpecificKeyData);

#endif


